/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2005 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "$Id: mx_lz.c,v 1.44.2.10 2006/09/24 02:09:35 loic Exp $";
#include "mx_arch.h"
#include "mx_instance.h"
#include "mx_malloc.h"
#include "mx_misc.h"
#include "mx_peer.h"
#include "mx_pio.h"
#include "mcp_config.h"
#include "mx_stbar.h"
#include "mx_util.h"
#include "ze_mcp_defs.h"
#include "mx_byteswap.h"
#include "mcp_global.h"
#include "mcp_printf.h"

static unsigned int
mx_ether_crc32 (const void *_ptr, unsigned int len)
{
  const char *ptr = (const char *)_ptr;
  unsigned int crc = 0xffffffff;
  int bitnum;
  
  for (bitnum=0; bitnum < len*8;bitnum++) {
    int bitval = (ptr[bitnum/8] >> (7-(bitnum&7))) & 1;
    int x32 = (crc >> 31) ^ bitval;
    crc = (crc << 1) ^ (x32 ? 0x04c11db7 : 0);
  }
  return ~crc;
}

#if !MX_HAS_PCIE_LINK_RESET
#define mx_pcie_link_reset(a) (-1)
#endif


/* enable interrupts */
static void
mx_lz_int_enable(mx_instance_state_t *is)
{
  if (mx_is_dead(is))
    return;
  MX_PIO_WRITE((uint32_t*)(is->lanai.sram + 0x740000),htonl(1));
  MX_STBAR();
}

/* disable interrupts */
static void
mx_lz_int_disable(mx_instance_state_t *is)
{
  if (mx_is_dead(is))
    return;
  MX_PIO_WRITE((uint32_t*)(is->lanai.sram + 0x740000),htonl(2));
  MX_STBAR();
  MX_READBAR();
  MX_PIO_READ((volatile uint32_t*)is->lanai.sram);
}

static void
mx_lz_park_board(mx_instance_state_t *is)
{
  struct mcp_print_info *msg;
  unsigned hdr_off;
  unsigned magic;
  uint16_t cmd;
  mx_read_pci_config_16(is, MX_PCI_COMMAND, &cmd);
  if (!is->bar0_low || !is->lanai.sram) {
    /* we never got far enough to try anything */
    return;
  }
  if (cmd & MX_PCI_COMMAND_MASTER) {
    hdr_off = ntohl(MX_PIO_READ((uint32_t *) (is->lanai.sram + 0x3c))) & 0xffffc;
    magic = ntohl(MX_PIO_READ((uint32_t *) (is->lanai.sram + hdr_off + 4)));
  } else {
    magic = -1;
  }
  if (mx_pcie_down == 0 &&  magic == 0x4d583130 /* MX10 */) {
    struct mx_lz_handoff req;
    MX_INFO((" handoff to eeprom\n"));
    req.bus_high = -1;
    req.bus_low = -1;
    req.data = htonl(0x12345678);
    req.sram_addr = -1;
    req.mcp_len = -1;
    req.to = -1;
    req.jump_addr = -1;
    mx_pio_memcpy((char*)is->lanai.sram + MX_LZ_HANDOFF_LOC, &req, sizeof(req), MX_PIO_FLUSH);
    mx_spin(500000);
    return;
  }
  mx_write_pci_config_16(is, 0x4 /*cmd*/, 2 /* memory */);
  mx_write_pci_config_32(is, 0x10, is->bar0_low);
  mx_write_pci_config_32(is, 0x14, is->bar0_high);
  mx_spin(1000);
  if (mx_pcie_down_on_error == 1 && mx_pcie_link_reset(is) != 0) 
    {
      /* if link reset is not available, we reboot the NIC by writing
	 a zero into the protected zone */
      is->lanai.sram[0] = 0x30;
      MX_STBAR();
    }
  mx_spin(1000000);
  mx_write_pci_config_32(is, 0x10, is->bar0_low);
  mx_write_pci_config_32(is, 0x14, is->bar0_high);
  mx_write_pci_config_32(is, 0x18, is->bar2_low);
  mx_write_pci_config_32(is, 0x1c, is->bar2_high);
  mx_write_pci_config_16(is, 0x4 /*cmd*/, 2 /* memory */);
  mx_write_pci_config_16(is, 0x46, is->msi_ctrl);
  mx_write_pci_config_32(is, 0X48, is->msi_addr_low);
  mx_write_pci_config_32(is, 0X4C, is->msi_addr_high);
  mx_write_pci_config_16(is, 0x50, is->msi_data);
  msg =  (void*)(is->lanai.sram + MCP_PRINT_OFF_ZE(0x1dfe00));
  if (MX_PIO_READ(&msg->magic) == htonl(MCP_PRINT_MAGIC)) {
    uint32_t len;
    uint32_t written;
    char *data;
    uint32_t buf_off;
    uint32_t i;

    buf_off = MX_PIO_READ(&msg->buf_off);
    buf_off = ntohl(buf_off);

    written = ntohl(MX_PIO_READ(&msg->written));
    len = written - is->ze_print_pos;
    is->ze_print_pos = written;
    if ((int)len < 0)
      return;
    len = MX_MIN(len, MCP_PRINT_BUF_SIZE);
    len = MX_MIN(len, 2048);
    if (!len)
      return;
    data = mx_kmalloc(len + 1, MX_WAITOK);
    if (!data)
      return;
    i = 0;
    while (len > 0) {
      data[i] = MX_PIO_READ((char*)is->lanai.sram 
			    + buf_off 
			    + ((written - len + i) & (MCP_PRINT_BUF_SIZE - 1)));
      if (data[i] == 0 || data[i] == '\n' || i == len) {
	data[i+1] = 0;
	MX_PRINT(("mx%d:Lanai: %s", is->id, data));
	len -= i + 1;
	i = 0;
      } else {
	i += 1;
      }
    }
    mx_kfree(data);
  }
}

static int
mx_lz_detect_parity(mx_instance_state_t *is)
{
  uint32_t reboot = -1;
  unsigned cap = mx_find_capability(is, MX_PCI_CAP_VENDOR);
  if (!cap) {
    MX_INFO(("Did device disappeard? Cannot get vendor cap\n"));
    return 0;
  }
  mx_write_pci_config_8(is, cap + 0x10, 3);
  mx_write_pci_config_32(is, cap + 0x18, 0xfffffff0);
  mx_read_pci_config_32(is, cap + 0x14, &reboot);
  MX_INFO(("REBOOT_STATUS=0x%x\n", reboot));
  return (reboot != -1) && (reboot & 0x3c000000);
}

static int
mx_lz_init_board(mx_instance_state_t *is)
{
  struct ze_hard_switch *hard_switch;
  int status;
  int mcp_len;
  void *img;
  volatile char *eeprom_strings;
  unsigned crc32, mmio_crc32;
  unsigned hdr_off, magic;
  int mcp_has_header;
  unsigned load_offset;

  mx_read_pci_config_32(is, 0x10, &is->bar0_low);
  mx_read_pci_config_32(is, 0x14, &is->bar0_high);
  mx_read_pci_config_32(is, 0x18, &is->bar2_low);
  mx_read_pci_config_32(is, 0x1c, &is->bar2_high);
  mx_read_pci_config_16(is, 0x46, &is->msi_ctrl);
  mx_read_pci_config_32(is, 0X48, &is->msi_addr_low);
  mx_read_pci_config_32(is, 0X4C, &is->msi_addr_high);
  mx_read_pci_config_16(is, 0x50, &is->msi_data);
  /* save the eeprom strings in case we overrite them by accident */
  eeprom_strings = (volatile char *)is->lanai.sram + MX_LZ_STRING_SPECS;
  bcopy((char *)eeprom_strings, is->lanai.eeprom_strings, 
	MX_EEPROM_STRINGS_LEN);
  is->lanai.eeprom_strings[MX_EEPROM_STRINGS_LEN - 1] = 0;
  is->lanai.eeprom_strings[MX_EEPROM_STRINGS_LEN - 2] = 0;

  mx_parse_eeprom_strings(is);

#if MX_OS_UDRV
  if (mx_lxgdb) {
    MX_WARN(("Assuming the mcp was directly loaded into lxgdb\n"));
    return 0;
  }
#endif
  hdr_off = ntohl(MX_PIO_READ((uint32_t *) (is->lanai.sram + 0x3c)));
  if (hdr_off == 0 || (hdr_off & 3) || hdr_off >= 1024*1024) {
    MX_INFO(("No MCP_GEN_HEADER on current firmware\n"));
    return EIO;
  }
  magic = MX_PIO_READ((uint32_t *) (is->lanai.sram + hdr_off + 4));
  if (magic != ntohl(0x70636965) /* "PCIE" */ &&
      magic != ntohl(0x45544820) /* "ETH " */) {
    char version[132];
    mx_pio_bcopy_read((void *)(is->lanai.sram + hdr_off + 4), version, 132);
    version[131] = 0;
    MX_INFO(("Did not find the std PCIE firmware, current(0x%x)=%s\n", ntohl(magic), version));
    return EIO;
  }

  
  MX_DEBUG_PRINT (MX_DEBUG_BOARD_INIT, ("Loading MCP\n"));
  mcp_len = 1024*1024;
  img = mx_kmalloc(mcp_len, MX_MZERO | MX_WAITOK);
  if (!img) {
    MX_WARN(("Could not allocate buffer to inflate mcp\n"));
    return ENOMEM;
  }
  status = mx_load_mcp(is, img, mcp_len, &mcp_len);
  if (status != 0) {
    MX_WARN(("mx%d: Could not load mcp\n", is->id));
    mx_kfree(img);
    return status;
  }
  mcp_has_header = ntohl(*(uint32_t*)((char*)img + 0x3c)) != 0;
  load_offset = mcp_has_header && !mx_pcie_down ? 0x110000 : ZE_HARD_SWITCH_OFFSET;
  if (mcp_len >= MX_LZ_STRING_SPECS - 32 * 1024 - load_offset) {
    MX_WARN(("MCP is too big: %d bytes, available for handoff = %d bytes\n",
	     mcp_len, MX_LZ_STRING_SPECS -32 * 1024 - load_offset));
    return E2BIG;
  }
  hard_switch = (void *)(is->lanai.sram + load_offset);
  mx_pio_memcpy(hard_switch->mcp, img, mcp_len, MX_PIO_32BIT_ALIGN | MX_PIO_32BYTE_FLUSH);
  MX_INFO(("Loaded mcp of len %d\n", mcp_len));
  crc32 = mx_ether_crc32(img, mcp_len);
  hard_switch->crc32 = htonl(crc32);
  hard_switch->magic = htonl(ZE_HARD_SWITCH_MAGIC);
  hard_switch->size = htonl(mcp_len);
  MX_READBAR();
  mx_pio_bcopy_read(hard_switch->mcp, img, mcp_len);
  mmio_crc32 = mx_ether_crc32(img, mcp_len);
  mx_kfree(img);

  if (hard_switch->crc32 != htonl(crc32) ||
      hard_switch->magic != htonl(ZE_HARD_SWITCH_MAGIC) ||
      hard_switch->size != htonl(mcp_len) ||
      mmio_crc32 != crc32) {
    MX_WARN(("Cannot switch firmware(crc=0x%x,len=%d)\n"
	     "\treread=(magic=0x%x,crc=0x%x,len=%d,crc_recomp=0x%x\n",
	     crc32, mcp_len, 
	     ntohl(hard_switch->magic), ntohl(hard_switch->crc32), 
	     ntohl(hard_switch->size), mmio_crc32));
    return EIO;
  }
  if (!mcp_has_header || mx_pcie_down) {
    /* hard hard_switch */
    uint32_t bar0, bar1, bar2, bar3, msi_addr0, msi_addr1;
    uint16_t command, msi_control, msi_data;
    uint32_t exp_devctl;

    /* save the important part of cfg space */
    mx_read_pci_config_32(is, 0x10, &bar0);
    mx_read_pci_config_32(is, 0x14, &bar1);
    mx_read_pci_config_32(is, 0x18, &bar2);
    mx_read_pci_config_32(is, 0x1c, &bar3);
    mx_read_pci_config_16(is, 0x4, &command);
    mx_read_pci_config_16(is, 0x46, &msi_control);
    mx_read_pci_config_32(is, 0X48, &msi_addr0);
    mx_read_pci_config_32(is, 0X4C, &msi_addr1);
    mx_read_pci_config_16(is, 0x50, &msi_data);
    mx_read_pci_config_32(is, 0X64, &exp_devctl);
    
    /* we reboot the NIC by writing a zero into the protected zone */
    is->lanai.sram[0] = 0;
    MX_STBAR();
    /* hope two and half second is enough to boot */
    mx_spin(2500000);

    
    /* restore cfg space */
    mx_write_pci_config_32(is, 0x10, bar0);
    mx_write_pci_config_32(is, 0x14, bar1);
    mx_write_pci_config_32(is, 0x18, bar2);
    mx_write_pci_config_32(is, 0x1c, bar3);
    mx_write_pci_config_16(is, 0x4, command);
    mx_write_pci_config_16(is, 0x46, msi_control);
    mx_write_pci_config_32(is, 0x48, msi_addr0);
    mx_write_pci_config_32(is, 0x4c, msi_addr1);
    mx_write_pci_config_16(is, 0x50, msi_data);
    mx_write_pci_config_32(is, 0X64, exp_devctl);
  } else {
    int i;
    uint16_t cmd;
    struct mx_lz_handoff req;
    struct ze_dump *dump;
    uint32_t dump_size = 512*1024;
    struct mcp_print_info *msg;

    hard_switch->magic = 0;

    dump = (struct ze_dump*)(is->lanai.sram + ZE_DUMP_OFF(MX_LZ_STRING_SPECS));
    dump->magic = 0;
    MX_STBAR();
    dump->addr = htonl(ZE_DUMP_JTAG_OFF(MX_LZ_STRING_SPECS) - dump_size);
    dump->pcie = htonl(ZE_DUMP_JTAG_OFF(MX_LZ_STRING_SPECS) - dump_size - 32 * 1024);
    dump->size = htonl(dump_size);
    MX_STBAR();
    dump->magic = htonl(ZE_DUMP_REQ);
    MX_STBAR();

    msg =  (void*)(is->lanai.sram + MCP_PRINT_OFF_ZE(0x1dfe00));
    if (MX_PIO_READ(&msg->magic) == htonl(MCP_PRINT_MAGIC))
      is->ze_print_pos = ntohl(MX_PIO_READ(&msg->written));

    /* enable busmaster DMA required for soft handoff */
    mx_read_pci_config_16(is, MX_PCI_COMMAND_MASTER, &cmd);
    mx_write_pci_config_16(is, MX_PCI_COMMAND_MASTER, cmd | MX_PCI_COMMAND_MASTER);
    *(uint32_t*)is->host_query.buf = 0;
    req.bus_high = htonl(is->host_query.pin.dma.high);
    req.bus_low = htonl(is->host_query.pin.dma.low);
    req.data = htonl(0x12345678);
    req.sram_addr = htonl(hard_switch->mcp - (char*)is->lanai.sram + 8);
    req.mcp_len = htonl(mcp_len - 8);
    req.to = htonl(8);
    req.jump_addr = htonl(8);
    mx_pio_memcpy((char*)is->lanai.sram + MX_LZ_HANDOFF_LOC, &req, sizeof(req), MX_PIO_FLUSH);
    /* 5 seconds timeout */
    for (i=0;; i++) {
      if (*(uint32_t*)is->host_query.buf == htonl(0x12345678)) {
	*(uint32_t*)is->host_query.buf = 0;
	break;
      }
      if (i > 50) {
	MX_INFO(("Handoff did not work, host flag=0x%x\n", 
		 *(uint32_t*)is->host_query.buf));
	return EIO;
      }
      mx_spin(100000);
    }
  }
  mx_globals(is)->z_loopback = htonl(mx_z_loopback);
  mx_lz_int_enable(is);
  return 0;
}

static void
mx_lz_claim_interrupt(mx_instance_state_t *is)
{
  MX_PIO_WRITE((uint32_t *) (is->lanai.sram + 0x700000), 0);
  MX_STBAR();
}

static int
mx_lz_might_be_interrupting(mx_instance_state_t *is)
{
  if (is->using_msi && !MX_CPU_powerpc64) {
    return MX_INTR_ACTIVE;
  } else {
    return MX_INTR_MAYBE;
  }
}

/* map the specials and SRAM.*/

static int
mx_lz_map_board(mx_instance_state_t *is)
{
  /* this is a temporary hack until I know how to interact with
     the mcp/bootloader */

  is->specials_size = 0;
  is->lanai.control_regs = NULL;
#ifdef MX_HAS_MAP_PCI_SPACE
  is->lanai.special_regs = mx_map_pci_space(is, 2, 0, 1024*1024);
  if (!is->lanai.special_regs)
    return ENOMEM;
#endif
  /* just pretend we have 2 MB of sram */
  if (MX_DEBUG)
    MX_INFO(("HACK: Pretending LZ has 2MB SRAM!\n"));

  is->sram_size = 2 * 1024 * 1024;

  is->lanai.sram = mx_map_pci_space(is, 0, 0, 16*1024*1024);
  if (is->lanai.sram == 0) {
    mx_unmap_io_space(is, 1*1024*1024, is->lanai.special_regs);
    return ENODEV;
  }

  return 0;
}

static void
mx_lz_unmap_board(mx_instance_state_t *is)
{
  mx_unmap_io_space(is, 16*1024*1024, (void*)is->lanai.sram);
  is->lanai.sram = NULL;
#ifdef MX_HAS_MAP_PCI_SPACE
  mx_unmap_io_space(is, 1*1024*1024, is->lanai.special_regs);
  is->lanai.special_regs = NULL;
#endif
}


static void
mx_lz_dump_registers(mx_instance_state_t *is,
		     uint32_t *reg, uint32_t *cnt)
{
  *cnt = 1;
  /* FIXME: not sure we can do for now, just fake ISR = 0 */
  if (reg) {
    reg[0] = 0x1f8;
    reg[1] = 0;
  }
}

static void
mx_lz_get_freq(mx_instance_state_t *is)
{
  int status;
  int exp_cap;
  uint16_t lnk_sta;
  
  status = mx_mcpi.get_param(is->id, is->lanai.sram, "clock_freq",
			     &is->cpu_freq);
  
  if (status || is->cpu_freq == 0) {
    MX_WARN(("mx%d: Failed to read LANai clock frequency (%d, %d)\n", 
	     is->id, status, is->cpu_freq));
    return;
  }
  exp_cap = mx_find_capability(is, MX_PCI_CAP_EXP);
  if (exp_cap) {
    mx_read_pci_config_16(is, exp_cap + MX_PCI_EXP_LNK_STA, &lnk_sta);
    is->pci_freq = MX_PCI_EXP_LNK_WIDTH(lnk_sta);
  }
}

static void
mx_lz_write_kreq(mx_instance_state_t *is, mcp_kreq_t *kreq)
{
  int kreqq_index;
  
  kreqq_index = is->kreqq_submitted & (is->kreqq_max_index);
  mx_always_assert((is->kreqq_submitted - is->kreqq_completed)
                   <= is->kreqq_max_index);
  is->kreqq_submitted++;
  mx_pio_memcpy(is->kreqq[kreqq_index].int64_array, kreq->int64_array, 
	      sizeof(mcp_kreq_t), MX_PIO_FLUSH);

  MX_PIO_WRITE((uint32_t *) (is->lanai.sram + 0x700004), 0);
  MX_STBAR();
}

mx_board_ops_t mx_lz_ops;

void
mx_lz_init_board_ops(void)
{
  mx_lz_ops.init		= mx_lz_init_board;
  mx_lz_ops.map			= mx_lz_map_board;
  mx_lz_ops.unmap		= mx_lz_unmap_board;
  mx_lz_ops.claim_interrupt	= mx_lz_claim_interrupt;
  mx_lz_ops.might_be_interrupting = mx_lz_might_be_interrupting;
  mx_lz_ops.enable_interrupt 	= mx_lz_int_enable;
  mx_lz_ops.disable_interrupt 	= mx_lz_int_disable;
  mx_lz_ops.park		= mx_lz_park_board;
  mx_lz_ops.detect_parity_error	= mx_lz_detect_parity;
  mx_lz_ops.dump_registers	= mx_lz_dump_registers;
  mx_lz_ops.get_freq		= mx_lz_get_freq;
  mx_lz_ops.write_kreq		= mx_lz_write_kreq;
}
